/*
 * ---------------------------------------------------------------------------
 * This is a slightly modified version of BINDVIEWS icmpenum which as its name 
 * suggests allows for the identification of hosts which are UP and respond 
 * to icmp messages of different types. BINDVIEWS original code (which is cool
 * BTW guys) did not however support ICMP MASK requests, which is very useful
 * especially when you have networks with Solaris - IOS - Windows :-) 
 *
 * Usage of this software is subject to BINDVIEWS license below.
 *
 * !Enjoy 
 *
 * Doc 
 * ---------------------------------------------------------------------------
 * icmpenum.c - host enumeration via icmp packets
 * 
 * Requirements: Libnet 1.0 or higher (http://www.packetfactory.net/libnet/)
 *               libpcap 0.4 (ftp://ftp.ee.lbl.gov/libpcap.tar.Z)
 *
 * Compilation instructions:
 *    gcc `libnet-config --defines` -o icmpenum icmpenum.c -lnet -lpcap
 *
 * Tested on RedHat 6.1/9.0 but should work most places that libnet and libpcap
 * compile and install properly.
 *
 * Initial revision 20Jan2000
 * Added Class C support and cleaned up the code 28Jan2000
Copyright (c) 2000 BindView Corporation.  All rights reserved.  

By using this software, YOU AGREE to the following license terms.  IF YOU DO NOT AGREE, YOU MAY NOT USE THE SOFTWARE.  

1.  BindView believes that this software is safe for use in normal circumstances, and has performed what it believes to be reasonable but non-exhaustive testing to verify this.  The software is intended for use only by experienced and knowledgeable computer professionals; IT IS PROVIDED "AS IS, WITH ALL FAULTS," including source code so that the user can study the source code and independently determine the software's suitability.  BindView makes no warranty of any kind, express or implied, and DISCLAIMS ANY AND ALL WARRANTIES, CONDITIONS, OR IMPLIED TERM OF QUALITY, INCLUDING THE IMPLIED WARRANTIES OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY, AND FITNESS FOR A PARTICULAR PURPOSE.  All use of the software is entirely at the user's own risk.

2.  IN NO EVENT WILL BINDVIEW BE LIABLE FOR DAMAGES OF ANY KIND arising from or relating to use of the software, whether such damages are direct, indirect, incidental, consequential, exemplary, or any other kind, and whether arising under contract, tort (including negligence), strict liability, or otherwise.

3.  BindView will not object to your distribution of complete, unmodified copies of the distribution package of the software as provided by BindView, PROVIDED that you do not charge a fee other than a reasonable fee for distribution services.  You may charge a fee for any warranty or support services that you offer to purchasers of copies of the software.

4.  You may modify the software and distribute copies of the modified software, PROVIDED:

	(a) that you distribute, together with the executable code of the modified software:
	
		(1) the source code of the modified software, which must contain the BindView copyright notice set forth above (in addition to your own copyright notice if any); and

		(2) a copy of the complete, unmodified distribution package of the software as provided by BindView; and

	(b) that you clearly indicate in the source code and in an accompanying documentation file that the software is based on BindView's software and was modified by you; and

	(c) that you grant users of the modified software the same rights as are granted to you by this license.

 */

#include <libnet.h>
#include <pcap.h> 
#include <net/bpf.h>

/*
 * Stolen from icmpquery. Chain structs to store host names.
 */
struct hosts
{
  char *hostname;
  u_long hostaddr;
  struct hosts *next;
};

struct hosts *hostnames;
struct hosts *hosttail;

/*
 * More globals
 */
struct bpf_program pcapfilter;
struct in_addr net, mask;
pcap_t *pd;
char pcap_err[PCAP_ERRBUF_SIZE];
u_char *ebuf;
char *dev;
struct libnet_link_int *l;
int num_of_targets, link_offset, verbose, targets_answered, unexpected;
u_short icmp_header;
int icmp_target;

/*
 * Build the targets
 */
void build_target(char *host, u_long ip_addr)
{
  if (hostnames == NULL)
  {
    hostnames = (struct hosts *) malloc(sizeof(*hostnames));
    if (hostnames == NULL)
    {
      perror("hostnames malloc failed");
      exit(-1);
    }
    hosttail = hostnames;
  } 
  else
  {
    hosttail->next = (struct hosts *) malloc(sizeof(*hostnames));
    if (hosttail->next == NULL) 
    {
      perror("hosttail->next malloc failed");
      exit(-1);
    }
    hosttail = hosttail->next;
  }
  hosttail->hostname = strdup(host);
  if (hosttail->hostname == NULL)
  {
    perror("strdup failed");
    exit(-1);
  }
  hosttail->hostaddr = ip_addr;
  hosttail->next = NULL;
}

/*
 * Build a class C of targets to ping
 */
void build_class_c(u_long dst_ip)
{
  int i, num_of_targets;
  u_long class_c_ips[255];
  union
  {
    struct in_addr addr;
    ulong temp_ip;
  } ip;
  
  for (i = 1; i < 255; i++)
  {
    class_c_ips[i] = htonl(dst_ip) + i;
    class_c_ips[i] = htonl(class_c_ips[i]);
    ip.temp_ip = class_c_ips[i];
    build_target(inet_ntoa(ip.addr),ip.temp_ip);
  }
}

/*
 * Process the targets, resolve the hosts, set up the chain, return the count
 */
int process_targets(char **hostlist)
{
  int i, num_of_targets=0;
  u_long tmpaddr;

  for (i = 0; hostlist[i]; i++)
  {
    tmpaddr = libnet_name_resolve(hostlist[i], LIBNET_RESOLVE);
    /* 
     * if a "local" hostname is given it could "resolve" to a 
     * broadcast address if the local host is not set up properly,
     * so we need to make sure we grab it and not send the packet 
     */
    if ((tmpaddr == 0xffffffff) || (tmpaddr == -1))
    {
      if (verbose) printf("%s not resolved\n",hostlist[i]);
      continue;
    };
    build_target(hostlist[i],tmpaddr);
    num_of_targets++;
  }
  return num_of_targets;
}

/*
 * build and send the icmp packets
 */
void send_packets(u_long src_ip)
{
  int packet_size, network, i;
  u_char *packet;
  u_long dst_ip;
  struct hosts *list;

  /* compute packet size */
  packet_size = LIBNET_IP_H + icmp_header;

  /* get mem for packet */
  libnet_init_packet(packet_size, &packet);
  if (packet == NULL)
  {
    libnet_error(LIBNET_ERR_FATAL, "unable to init packet mem\n");
  }

  /* get network ready */
  network = libnet_open_raw_sock(IPPROTO_RAW);
  if (network == -1)
  {
    libnet_error(LIBNET_ERR_FATAL, "unable to open network for sending\n");
  }

  list = hostnames;

  /* main sending loop */
  while (list != NULL)
  {
    /* get the target address */
    dst_ip = list->hostaddr;

    /* build IP section */
    libnet_build_ip(LIBNET_ICMP_TS_H,0,666,0,255,IPPROTO_ICMP,src_ip,dst_ip,NULL,0,packet);
    /* build ICMP section */
    switch (icmp_target)
    {
      case ICMP_ECHOREPLY:
        libnet_build_icmp_echo(ICMP_ECHO,0,666,0,NULL,0,packet + LIBNET_IP_H);
        break;        
      case ICMP_TSTAMPREPLY:
        /*
         * note that we leave the actual timestamp values blank, I
         * could do that but since we are doing host enumeration I
         * figured I could leave out all the "seconds from midnight"
         * code -- less stuff to <include>, I guess 
         */
        libnet_build_icmp_timestamp(ICMP_TSTAMP,0,666,0,0x00000000,0x00000000,0x00000000,NULL,0,packet + LIBNET_IP_H);
        break;
      case ICMP_IREQREPLY:
        /* 
         * since libnet doesn't do info request/replies, we cheat
         * since the header for echo is the same size as info
         */
        libnet_build_icmp_echo(ICMP_IREQ,0,666,0,NULL,0,packet + LIBNET_IP_H);
        break;        
      case ICMP_MASKREPLY:
	libnet_build_icmp_mask(ICMP_MASKREQ,0,666,0,0,NULL,0xfffffff0,packet + LIBNET_IP_H);
        break;

        /*  Doc: Wanted to build in MASK Requests, as many systems like
         *  Solaris/IOS/Windows often leave support for this in place.
         */
        libnet_build_icmp_mask(ICMP_MASKREQ,0,242,0,0xffffffff,NULL,0,packet + LIBNET_IP_H);
        break;
 

   }
    /* do the checksum */
    if (libnet_do_checksum(packet, IPPROTO_ICMP, icmp_header) == -1) 
    { 
      libnet_error(LIBNET_ERR_FATAL, "checksum failed on %s\n", list->hostname); 
    }

    /* send the packet */
    i = libnet_write_ip(network, packet, packet_size);
 
    if (i == -1)
    {
      libnet_error(LIBNET_ERR_FATAL, "failed to write to network\n");
    }

    /* 
     * this isn't *too* bad, but if we're trying to be stealth there
     * is no reason to send part of a packet in case it does get there
     * as it might set off an alarm. therefore if we start trying to
     * send just parts, abandon ship 
     */
    if (i < packet_size) 
    { 
      libnet_error(LIBNET_ERR_FATAL, "only wrote %d bytes\n", i); 
    }

    if (verbose)
    { 
      printf("Sent %d bytes for %s\n", i, list->hostname); 
    }

    /* 
     * miniscule delay between packets to prevent overflowing the local
     * interface
     */ 
    usleep(1);

    /* get next address, or NULL if done */
    list = list->next;
  }
  
  /* clean things up */
  if (libnet_close_raw_sock(network) == -1) 
  { 
    libnet_error(LIBNET_ERR_WARNING, "couldn't close the interface after sending"); 
  } 
  libnet_destroy_packet(&packet); 
}

/*
 * Receive the packets normally, i.e. no sniffing involved.
 */
void receive_packets_normal(void)
{
  char buf[1500];
  struct ip *ip = (struct ip *)buf;
  struct icmp *icmp = (struct icmp *)(ip + 1);
  int s, err = 0, on = 1;
  long int fromlen = 0;
  ulong src_ip;
  struct hosts *targets;

  targets_answered = 0;
  unexpected = 0;

  /* for non-sniffing we listen the old-fashioned way */
  if ((s = socket(AF_INET, SOCK_RAW, IPPROTO_ICMP)) < 0) 
  {
    fprintf(stderr, "Problem opening socket for listening");
    exit(-1);
  }

  if (setsockopt(s, IPPROTO_IP, IP_HDRINCL, &on, sizeof(on)) < 0) 
  {
    fprintf(stderr, "Problem setting option on socket");
    exit(-1);
  }

  while (targets_answered < num_of_targets)
  {
    if ((err = recvfrom(s, buf, sizeof buf, 0, NULL, (int *)&fromlen)) < 0)
    {
      fprintf(stderr,"error receiving on network");
    }

    targets = hostnames; /* reload fresh before each address compare */
    while (targets != NULL)
    {
      if (targets->hostaddr == ip->ip_src.s_addr) 
      {
        if ((IPPROTO_ICMP == ip->ip_p) && (icmp_target == icmp->icmp_type))
        {
          fprintf(stdout,"%s is up\n", inet_ntoa(ip->ip_src));
          targets_answered++;
        }
        else
        {
          /* we do not wish to see any odd packets from our own address */
          src_ip = libnet_get_ipaddr(l, dev, ebuf);
          src_ip = htonl(src_ip);
          if (!(src_ip == ip->ip_src.s_addr))
          {  
            if (verbose) printf("Received unexpected ICMP packet from %s\n",inet_ntoa(ip->ip_src));
            unexpected++;
          }
        }
      }
      targets=targets->next;
    }
  }
}

/*
 * this handles packets as they are sniffed and processes them, similar to
 * the receive_packets_normal routine
 */
void grab_icmp(u_char *data1, struct pcap_pkthdr* h, u_char *p)
{
  struct ip* ip = (struct ip *)(p + link_offset);
  struct icmp *icmp = (struct icmp *)(ip + 1);
  struct hosts *targets;

  targets = hostnames;
  while (targets != NULL)
  {
    if (targets->hostaddr == ip->ip_src.s_addr) 
    {
      if ((IPPROTO_ICMP == ip->ip_p) && (icmp_target == icmp->icmp_type))
      {
        fprintf(stdout,"%s is up\n", inet_ntoa(ip->ip_src));
        targets_answered++;
	}
{
	fprintf(stdout,"Response Type was %s \n",(icmp->icmp_type));
} 
     } 
      else
      {
        if (verbose) printf("Received unexpected ICMP packet from %s\n",inet_ntoa(ip->ip_src));
        unexpected++;
      }
    }
    targets=targets->next;
  }


/*
 * So why do a "normal" and a "sniffing" routine separately? Sniffing requires
 * root and putting the receiving interface in promiscuous mode -- that and I
 * simply never removed the normal routine and decided to leave it. This is a
 * proof of concept tool, not proof of logic ;-) 
 */
void receive_packets_sniffing(void)
{
  int snaplen = 65535, promisc = 1, to_ms = 1000;

  if (pcap_lookupnet(dev,&net.s_addr,&mask.s_addr, pcap_err) == -1) 
  {
    perror(pcap_err);
    exit(-1);
  } 

  if (verbose) printf("listening for replies on %s: %s\n", dev, inet_ntoa(net));

  if ((pd = pcap_open_live(dev, snaplen, promisc, to_ms, pcap_err)) == NULL)
  {
    perror(pcap_err);
    exit(-1);
  }

  switch(pcap_datalink(pd)) 
  {
    case DLT_EN10MB:
    case DLT_IEEE802:
      link_offset = 14;
      break;
    case DLT_SLIP: 
      link_offset = 16;
      break;
    case DLT_PPP:
    case DLT_NULL:
      link_offset = 4;
      break;
    case DLT_RAW: 
      link_offset = 0;
      break;
    default:
      fprintf(stderr,"unsupported interface type\n");
      exit(-1);
  }

  while (pcap_loop(pd,0,(pcap_handler)grab_icmp,0));
}

/*
 * Called by caughtsig, if we're sniffing to turn off the promisc bit
 */
void stop_sniffing(void)
{
  int fd, i;
  struct ifreq buf[16];
  struct ifconf ifc;

  if ((fd = socket (AF_INET, SOCK_DGRAM, 0)) >= 0)
  {
    ifc.ifc_len = sizeof buf;
    ifc.ifc_buf = (caddr_t) buf;
    if (!(ioctl (fd, SIOCGIFCONF, (char *) &ifc))) 
    {
      i = ifc.ifc_len / sizeof (struct ifreq);
      if (!i)
      {
        fprintf(stderr, "Can't find an interface.\n");
        exit(-1);
      }
      while (i-- > 0)
      {
        /* cmp only length of dev since ifr_name may have extra chars */
        if (!(strncmp(buf[i].ifr_name,dev,strlen(dev))))
        {
          if(!(ioctl(fd, SIOCGIFFLAGS, (char *) &buf[i])))
          {
            if (buf[i].ifr_flags & IFF_PROMISC) 
            {
              buf[i].ifr_flags = buf[i].ifr_flags ^ IFF_PROMISC;
              if(ioctl(fd, SIOCSIFFLAGS, &buf[i])) perror("ioctl");
            }
          }
          else perror("ioctl");
        } 
      } /* end while loop */
    } 
    else perror ("ioctl");
  } 
  else perror ("socket");
  close (fd);
}

void totals(void)
{
  printf("Total expected packets received   : %d\n",targets_answered);
  printf("Total unexpected packets received : %d\n",unexpected);
}

void caughtsig(int sig)
{
  if (pd) stop_sniffing();
  if (verbose) totals();
  exit(sig);
}

/*
 * usage
 */
void usage(char *prog)
{
  fprintf(stderr,"\nThis is Version 1.2 of icmpenum, Enjoy!\n\n");
  fprintf(stderr,"USAGE: ");
  fprintf(stderr,"%s [opts] [-c class C] [-d dev] [-i 1-3] [-s src] [-t sec] hosts\n\n",prog);
  fprintf(stderr,"\nExample: icmpenum -i 4 -c 1.2.3.4 (scans class C 1.2.3.1/24 for ICMP type 18`s\n\n");
  fprintf(stderr,"  opts are h n p r v\n");
  fprintf(stderr,"    -h this help screen\n");
  fprintf(stderr,"    -n no sending of packets\n");
  fprintf(stderr,"    -p promiscuous receive mode\n");
  fprintf(stderr,"    -r receiving packets only (no\n");
  fprintf(stderr,"    -v verbose\n");
  fprintf(stderr,"  -c class C in x.x.x.0 form\n");
  fprintf(stderr,"  -i icmp type to send/receive, types include the following:\n");
  fprintf(stderr,"       1  echo/echo reply (default)\n");
  fprintf(stderr,"       2  timestamp request/reply\n");
  fprintf(stderr,"       3  info request/reply\n");
  fprintf(stderr,"       4  mask request/reply\n");
  fprintf(stderr,"  -d device to grab local IP or sniff from, default is eth0\n");
  fprintf(stderr,"  -s spoofed source address\n");
  fprintf(stderr,"  -t time in seconds to wait for all replies (default 5)\n");
  fprintf(stderr,"  host(s) are target hosts (ignored if using -c)\n");
  fprintf(stderr,"\n");
}

int main(int argc, char **argv)
{
  char *prog;
  extern char *optarg;
  extern int optind;
  extern int optopt;
  extern int opterr;
  char ch;
  int spoof = 0, receive = 0, promiscuous = 0, no_send = 0, setdev = 0;
  int sent, timeout = 5, icmpenum = 1, class_c = 0;
  u_long src_ip;
  u_long dst_ip = 0;
  u_long class_c_ip;
  pid_t pid;
  u_long CLASS_C_MASK = 0x00ffffff;
  
  dev = "eth0"; /* default */
  verbose = 0;
  prog = argv[0];

  while ((ch = getopt(argc, argv, "hrnpvc:d:i:s:t:")) != EOF) 
    switch(ch)
    {
      case 'c':
        dst_ip = libnet_name_resolve(optarg, LIBNET_RESOLVE);
        if (dst_ip == -1)
        {
          printf("=== Invalid class C %s\n", optarg);
          usage(prog);
          exit(-1);
        }
        /* get the last part of the address */
        class_c_ip = ((dst_ip | CLASS_C_MASK) - CLASS_C_MASK);
        /* if it is not zero make it zero */
        if(class_c_ip) dst_ip = dst_ip ^ class_c_ip;
        class_c = 1;
        break;
      case 'h':
        usage(prog);
        exit(0);
      case 'i':
        icmpenum = (int) strtol(optarg, NULL, 10);
        break;
      case 'r':
        receive = 1;
        break;
      case 'p':
        promiscuous = 1;
        break;
      case 'n':
        no_send = 1;
        break;
      case 'd':
        dev = optarg;
        setdev = 1;
        break;
      case 's':
        if (!(src_ip = libnet_name_resolve(optarg, LIBNET_RESOLVE)))
        {
          fprintf(stderr,"=== Unable to resolve source host,\n");
          fprintf(stderr,"=== try spoofing with an IP address\n");
          usage(prog);
          exit(-1);
        }
        spoof = 1;
        break;
      case 't':
        timeout = (int) strtol(optarg, NULL, 10);
        break;
      case 'v':
        verbose = 1;
        break;
      default:
        usage(prog);
        exit(-1);
    }
  argc -= optind;
  argv += optind;

  /* post arg processing */
  switch(icmpenum)
  {
    case 0:
    case 1:
      icmp_header = LIBNET_ICMP_ECHO_H;
      icmp_target = ICMP_ECHOREPLY;
      break;
    case 2:
      icmp_header = LIBNET_ICMP_TS_H;
      icmp_target = ICMP_TSTAMPREPLY;
      break;
    case 3:
      /* 
       * since libnet doesn't do info request/replies, we cheat
       * since the header for echo is the same size
       */
      icmp_header = LIBNET_ICMP_ECHO_H;
      icmp_target = ICMP_IREQREPLY;
      break;
    case 4:
      icmp_header = LIBNET_ICMP_TS_H;
      icmp_target = ICMP_MASKREPLY;
      break;
    
    default:
      fprintf(stderr,"=== Invalid or unsupported icmp type\n");
      usage(prog);
      exit(-1);
  }

  if ((setdev) && (spoof))
  {
    fprintf(stderr, "=== You cannot specify a device for a source IP address\n");
    fprintf(stderr, "=== and spoof your source IP address at the same time.\n");
    usage(prog);
    exit(-1);
  }

  if ((no_send) && (!promiscuous) && (!receive))
  {
    fprintf(stderr,"=== You are not sending and receiving any packets,\n");
    fprintf(stderr,"=== so not much is going to happen.\n");
    usage(prog);
    exit(-1);
  }

  /* if the receive flag is set, set the no_send flag so we don't send
     any packets */
  if ((receive) && (!no_send)) no_send++;

  if ((spoof) && (!receive) && (!promiscuous))
  {
    printf("NOTE: This machine will not see responses while spoofing.\n");
    timeout = 0; /* no need to wait around for replies */
  }
  else
  {
    src_ip = libnet_get_ipaddr(l, dev, ebuf);
    if ((src_ip == -1) || (src_ip == 0))
    {
      fprintf(stderr, "=== Grabbing address from %s failed,\n",dev);
      fprintf(stderr, "=== try a different device.\n");
      usage(prog);
      exit(-1);
    }
    src_ip = htonl(src_ip);
  }  

  if ((receive) && (promiscuous))
  {
    fprintf(stderr,"=== Receive mode is for normal receiving of spoofed packets\n");
    fprintf(stderr,"=== while Promiscuous mode is for sniffing of spoofed packets\n");
    fprintf(stderr,"=== passing nearby. They cannot be used together.\n");
    usage(prog);
    exit(-1);
  }

  if ((!class_c) && (!argv[0] || !strlen(argv[0])))
  {
    fprintf(stderr,"=== You must specify target(s) or a class C to enumerate\n");
    usage(prog);
    exit(-1);
  }

  /* end post arg processing */

  if (class_c)
  /* build a class C full of addresses */
  {
    if (timeout < 10) timeout = timeout + 10;
    build_class_c(dst_ip);
    num_of_targets = 255;
  }
  /* build target chain and get a count */
  else 
  {
    num_of_targets = process_targets(argv);
  }

  /* set up timeout*/
  signal(SIGALRM, caughtsig);
  alarm(timeout);

  /* if sending packets, fork off the process so we can start listening
     to replies immediately */
  pid = fork();
  switch(pid)
  {
    case -1:
      /* if not sending we don't care that much about fork failures */
      if (!no_send)
      {
        fprintf(stderr, "Unable to fork off for sending packets\n");
        exit(-1);
      }
      break;
    case 0:
      if (!no_send) send_packets(src_ip);
      break;
    default:
      if (!spoof)
      {
        receive_packets_normal();
        if (verbose) totals();
        break;
      }
      if (promiscuous)
      {
        receive_packets_sniffing();
        break;
      }
  }
  exit(0);
}
